package app.firezone.synk.network.Activities;

import android.annotation.TargetApi;
import android.app.AlertDialog;
import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;

import androidx.appcompat.app.AppCompatActivity;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;

import app.firezone.synk.network.API.SharedPrefManager;
import app.firezone.synk.network.MainActivity;
import app.firezone.synk.network.R;

public class SplashActivity extends AppCompatActivity {

    private final Handler handler = new Handler(Looper.getMainLooper());

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_splash);

        checkConnectionAndProceed();
    }

    private void checkConnectionAndProceed() {
        new Thread(() -> {
            boolean connected = isInternetAvailable() && hasRealInternetAccess();
            handler.post(() -> {
                if (!connected) {
                    startActivity(new Intent(SplashActivity.this, NoInternetActivity.class));
                    finish();
                } else {
                    showSlowInternetDialogIfNeeded();
                }
            });
        }).start();
    }

    private void showSlowInternetDialogIfNeeded() {
        new Thread(() -> {
            boolean isSlow = isInternetSlow();
            handler.post(() -> {
                if (isSlow) {
                    showSlowDialog();
                } else {
                    goToNextScreenWithDelay();
                }
            });
        }).start();
    }

    private void showSlowDialog() {
        new AlertDialog.Builder(SplashActivity.this)
                .setTitle("Slow Internet")
                .setMessage("Your internet connection seems slow. Please wait or switch to a better network.")
                .setCancelable(false)
                .setPositiveButton("Retry", (dialog, which) -> checkConnectionAndProceed())
                .setNegativeButton("Exit", (dialog, which) -> finish())
                .show();
    }

    private void goToNextScreenWithDelay() {
        handler.postDelayed(() -> {
            long userId = SharedPrefManager.getInstance(SplashActivity.this).getUserId();
            Intent intent = (userId == -1)
                    ? new Intent(SplashActivity.this, LoginActivity.class)
                    : new Intent(SplashActivity.this, MainActivity.class);
            startActivity(intent);
            finish();
        }, 1000); // 1 second delay for smoother UX
    }

    private boolean isInternetAvailable() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            return isInternetAvailableModern();
        } else {
            return isInternetAvailableLegacy();
        }
    }

    @TargetApi(Build.VERSION_CODES.M)
    private boolean isInternetAvailableModern() {
        ConnectivityManager cm = (ConnectivityManager) getSystemService(Context.CONNECTIVITY_SERVICE);
        if (cm == null) return false;

        Network network = cm.getActiveNetwork();
        if (network == null) return false;

        NetworkCapabilities capabilities = cm.getNetworkCapabilities(network);
        return capabilities != null &&
                capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) &&
                capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED);
    }

    @SuppressWarnings("deprecation")
    private boolean isInternetAvailableLegacy() {
        ConnectivityManager cm = (ConnectivityManager) getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetwork = cm != null ? cm.getActiveNetworkInfo() : null;
        return activeNetwork != null && activeNetwork.isConnected();
    }

    private boolean hasRealInternetAccess() {
        try {
            HttpURLConnection urlc = (HttpURLConnection)
                    (new URL("https://clients3.google.com/generate_204").openConnection());
            urlc.setRequestProperty("User-Agent", "Android");
            urlc.setRequestProperty("Connection", "close");
            urlc.setConnectTimeout(1500);
            urlc.connect();
            return (urlc.getResponseCode() == 204);
        } catch (IOException e) {
            return false;
        }
    }

    private boolean isInternetSlow() {
        try {
            long start = System.currentTimeMillis();
            HttpURLConnection connection = (HttpURLConnection) new URL("https://clients3.google.com/generate_204").openConnection();
            connection.setConnectTimeout(2000);
            connection.connect();
            long end = System.currentTimeMillis();
            return (end - start) > 2000;
        } catch (IOException e) {
            return true;
        }
    }

    @Override
    public void onBackPressed() {
        // Optionally disable back press during splash
        super.onBackPressed();
    }
}
